<?php
class ChatMessage {
    // Database connection
    private $conn;
    private $table = "chat_messages";

    // Properties
    public $chat_msg_id;
    public $user_id;
    public $content;
    public $created_at;
    public $location_id;
    public $company_id;

    // Constructor
    public function __construct($db) {
        $this->conn = $db;
    }

    public function readMessagesBetween($loggedInUserId, $selectedEntityId, $entityType) {
        $query = "";

        switch ($entityType) {
            case "user":
                $query = "SELECT * FROM " . $this->table . " WHERE (user_id = ? AND location_id = ?) OR (user_id = ? AND company_id = ?) ORDER BY created_at ASC";
                break;
            case "location":
                $query = "SELECT * FROM " . $this->table . " WHERE (location_id = ? AND user_id = ?) OR (location_id = ? AND company_id = ?) ORDER BY created_at ASC";
                break;
            case "company":
                $query = "SELECT * FROM " . $this->table . " WHERE (company_id = ? AND user_id = ?) OR (company_id = ? AND location_id = ?) ORDER BY created_at ASC";
                break;
            default:
                return [];
        }

        $statement = $this->conn->prepare($query);
        $statement->bindValue(1, $loggedInUserId, PDO::PARAM_INT);
        $statement->bindValue(2, $selectedEntityId, PDO::PARAM_INT);
        $statement->bindValue(3, $selectedEntityId, PDO::PARAM_INT);
        $statement->bindValue(4, $loggedInUserId, PDO::PARAM_INT);

        $statement->execute();

        $result = $statement->fetchAll(PDO::FETCH_ASSOC);

        return $result;
    }

    // Get messages based on user, company, or location ID
    public function read($userId, $companyId, $locationId) {
        if (!empty($userId)) {
            // User is logged in with a personal account
            $query = "SELECT * FROM " . $this->table . " WHERE user_id = ? ORDER BY created_at ASC";
            $statement = $this->conn->prepare($query);
            $statement->bindValue(1, $userId, PDO::PARAM_INT);
        } elseif (!empty($companyId)) {
            // User is logged in with a company account
            $query = "SELECT * FROM " . $this->table . " WHERE company_id = ? ORDER BY created_at ASC";
            $statement = $this->conn->prepare($query);
            $statement->bindValue(1, $companyId, PDO::PARAM_INT);
        } elseif (!empty($locationId)) {
            // User is logged in with a location account
            $query = "SELECT * FROM " . $this->table . " WHERE location_id = ? ORDER BY created_at ASC";
            $statement = $this->conn->prepare($query);
            $statement->bindValue(1, $locationId, PDO::PARAM_INT);
        } else {
           
            return [];
        }

        $statement->execute();

        $result = $statement->fetchAll(PDO::FETCH_ASSOC);

        return $result;
    }

    // Get sender's name
    public function getSenderName($userId, $companyId, $locationId) {
        if (!empty($userId)) {
            // User is logged in with a personal account
            $query = "SELECT user_fname, user_lname FROM users WHERE user_id = ?";
            $statement = $this->conn->prepare($query);
            $statement->bindValue(1, $userId, PDO::PARAM_INT);
        } elseif (!empty($companyId)) {
            // User is logged in with a company account
            $query = "SELECT company_name FROM companies WHERE company_id = ?";
            $statement = $this->conn->prepare($query);
            $statement->bindValue(1, $companyId, PDO::PARAM_INT);
        } elseif (!empty($locationId)) {
            // User is logged in with a location account
            $query = "SELECT
                        CONCAT(companies.company_name, '-', locations.location_name) AS name
                        FROM locations 
                        INNER JOIN companies ON locations.company_id = companies.company_id
                        WHERE location_id = ?";
            $statement = $this->conn->prepare($query);
            $statement->bindValue(1, $locationId, PDO::PARAM_INT);
        } else {
           
            return "Unknown Sender";
        }

        $statement->execute();
        $result = $statement->fetch(PDO::FETCH_ASSOC);

        // Combine first and last name or get company name or location name
        return (!empty($result)) ? implode(' ', $result) : "Unknown Sender";
    }
    
    // Create 
    public function create() {
        $query = "INSERT INTO " . $this->table . "
        SET
        user_id = :user_id,
        content = :content,
        created_at = CURRENT_TIMESTAMP,
        location_id = :location_id,
        company_id = :company_id
        ";
    
        $stmt = $this->conn->prepare($query);
    
        // Filter data
        $this->user_id = htmlspecialchars(strip_tags($this->user_id));
        $this->content = htmlspecialchars(strip_tags($this->content));
        $this->location_id = htmlspecialchars(strip_tags($this->location_id));
        $this->company_id = htmlspecialchars(strip_tags($this->company_id));
    
        // If user_id is not provided, set it to NULL
        $userIdParam = !empty($this->user_id) ? $this->user_id : null;
    
        // If company_id is not provided, set it to NULL
        $companyIdParam = !empty($this->company_id) ? $this->company_id : null;
    
        // If location_id is not provided, set it to NULL
        $locationIdParam = !empty($this->location_id) ? $this->location_id : null;
    
        // Bind data
        $stmt->bindParam(":user_id", $userIdParam, PDO::PARAM_INT);
        $stmt->bindParam(":content", $this->content);
        $stmt->bindParam(":location_id", $locationIdParam, PDO::PARAM_INT);
        $stmt->bindParam(":company_id", $companyIdParam, PDO::PARAM_INT);
    
        // Execute
        if ($stmt->execute()) {
            return true;
        }
    
        printf("Error: %s", $stmt->error);
        return false;
    }
    
    // // Update 
    // public function update() {
    //     $query = "UPDATE " . $this->table . "
    //     SET
    //         user_id = :user_id,
    //         content = :content,
    //         created_at = :created_at,
    //         location_id = :location_id,
    //         company_id = :company_id
    //     WHERE
    //         chat_msg_id = :chat_msg_id
    //     ";

    //     $stmt = $this->conn->prepare($query);

    //     // Filter data
    //     $this->user_id = htmlspecialchars(strip_tags($this->user_id));
    //     $this->content = htmlspecialchars(strip_tags($this->content));
    //     $this->created_at = htmlspecialchars(strip_tags($this->created_at));
    //     $this->location_id = htmlspecialchars(strip_tags($this->location_id));
    //     $this->company_id = htmlspecialchars(strip_tags($this->company_id));
    //     $this->chat_msg_id = htmlspecialchars(strip_tags($this->chat_msg_id));

    //     // Bind data
    //     $stmt->bindParam(":user_id", $this->user_id);
    //     $stmt->bindParam(":content", $this->content);
    //     $stmt->bindParam(":created_at", $this->created_at);
    //     $stmt->bindParam(":location_id", $this->location_id);
    //     $stmt->bindParam(":company_id", $this->company_id);
    //     $stmt->bindParam(":chat_msg_id", $this->chat_msg_id);

    //     // Execute
    //     if ($stmt->execute()) {
    //         return true;
    //     }

    //     printf("Error: %s", $stmt->error);
    //     return false;
    // }
    
   
}